/*
MedImage Wound Analysis Add-on
==============================

Developed by (C) AtomJump Ltd. 9 Aug 2017. This is a commercial add-on to MedImage, and is licensed
by AtomJump on a per-server basis.


This script is run from the MedImage server, or independently as
node wound-size.js photo/path/file.jpg

and this photo should include a sticker and a wound. Preferably do not include background colours
that match the sticker colour in the photo, as this could result in a non-detection of the sticker
and a resulting mismatch of the area of the wound. The input path is relative to the config/master.json
photosRootPath parameter.

It will output 
photo/path/file.json
and
photo/path/file.wound-view.jpg

The .json is a descriptive file with the results for that photo, including the size in square cm of the wound.
The wound itself may be ambiguous between several different parts of the photo. These will be listed as 
ambiguous options in the json, and by default, the closest to the centre of the photo is used.  
However, the .json can be modified to include a different centre by modifying input.selectedWoundPoint, and 
then this script is run again.

The .wound-view.jpg file, is a smaller version of the photo, with the highlighted sticker and wound. Other
ambiguous options for the wound are also displayed in a different colour. This allows a different script to 
allow user feedback on which wound is the correct one, by selecting a point in this smaller photo, and then
modifying input.selectedWoundPoint (which will be in the original scale).

Typically, all the colour ranges and other parameters will initially come from the config/master.json file,
but they can be modified by hand on an individual photo basis for finer control.



Installation
============

Firstly, you will need nodejs and opencv installed.

For opencv, as mentioned at http://milq.github.io/install-opencv-ubuntu-debian/

sudo apt-get install libopencv-dev python-opencv
will work on Ubuntu, and this is adequate. That doc also mentions the offical route.

Then, as mentioned by https://www.npmjs.com/package/opencv

npm install opencv
npm install upath

However, Windows is the exception. You will need a valid license (the free community edition is potentially possible)
for Visual Studio. This package costs around US$500 for a professional version.




Research
========

https://stackoverflow.com/questions/29156091/opencv-edge-border-detection-based-on-color

See
https://community.risingstack.com/opencv-tutorial-computer-vision-with-node-js/
for a tutorial on NodeJS OpenCV

and
https://github.com/peterbraden/node-opencv/wiki/Matrix
for commands in NodeJS

For an HSV colour editor see:
http://colorizer.org

First attempts:

rgb of blister
185 105 97
= 5 38 55  in HSV (0-360, 0-100, 0-100)

Opencv uses HSV 0-180, 0-255, 0-255
So blister = Open CV HSV 2 96 140 

rgb of ordinary skin
200 150 139
= 10 35 66  in HSV (0-360, 0-100, 0-100)
So skin = Open CV HSV 5 89 168

Also see https://www.researchgate.net/publication/259644745_Segmentation_of_Chronic_Wound_Areas_by_Clustering_Techniques_Using_Selected_Color_Space


Wound colour ranges

s = 37 - 100
v = 14 - 60
	
in OpenCV HSV: s = 94 - 255
	           v = 35 - 154
	
Sticker, or an NZ 50c coin:
radius of 50c coin 2.4 cm. Area = 4.52389 square cm


Search google for 'area of circle', for quick sizing calculator. Use more details for 4dp.


*/



var jimp = require('jimp');
var exec = require('child_process').exec;
var spawn = require('child_process').spawn;


//To allow for larger images, and hopefully speed up the image read. See https://github.com/jimp-dev/jimp/issues/915
/*const cachedJpegDecoder = jimp.decoders['image/jpeg'];
jimp.decoders['image/jpeg'] = (data) => {
  const userOpts = { maxMemoryUsageInMB: 1024 }
  return cachedJpegDecoder(data, userOpts)
}*/

var fs = require('fs');
var upath = require('upath');
var path = require('path');
var queryString = require('querystring');
var resolve = require('path').resolve;
var fsExtra = require('fs-extra');
var area = require('area-polygon');


var masterConfigFile = __dirname + '/config/master.json';
var mainMedImagePath = "../../photos/";
var mainMedImagePathAfterBackup = "../../photos/";
var globalConfigFile = __dirname + '/../../config.json';
global.mainMedImagePathAfterBackup = mainMedImagePathAfterBackup;

var verbose = false;			//Should be 'false' before a release
var webForm = false;		//Whether we came from a web form, or on the command line



//Include shared funcs
var normalizeInclWinNetworks = require(__dirname + '/common/libfuncs.js').normalizeInclWinNetworks;
var readConfig = require(__dirname + '/common/libfuncs.js').readConfig;
var writeConfig = require(__dirname + '/common/libfuncs.js').writeConfig;

//Include area functions
var refresh = require(__dirname + '/common/refresh-thumbnail.js'); 
 



function getUserDrawings(conf, img, width, height)
{
	//Returns contours object based off an array of points drawn by a human with a mouse or on a tablet.
	//The coordinates are in an array format ie.  [[x1,y1],[x2,y2],[x3,y3] ... ], and are already sitting in
	//the config file conf, in the correct pixel scales
	if(verbose == true) console.log("Getting user drawings");
	var WHITE = [255];	
	var BLACK = [ 0];
	var lineWidth = 8;
	var newSketches = [];
	
	if(conf.input.userSketch) {
	
		//Create a buffer matrix
		var bufferMat = img.copy();
		bufferMat.convertGrayscale();
		bufferMat.rectangle([0,0],[width,height], BLACK, -1);	
		
		if(verbose == true) console.log("Drawings exist, created buff");
		
		var sketch = conf.input.userSketch;
		
		for(var cnt=0; cnt< sketch.length; cnt++) {
			//Loop through all of the points	
			
			if(verbose == true) console.log("Using drawing " + cnt);
			if(sketch[cnt].newSketch == true) {
				sketch[cnt].newSketch = false;		//Switch off any chance of this being considered new next time
				newSketches.push(cnt);		//It is likely that this id number for this area, should be selected by default			
											//Note: this is not precise, and if there are overlapping user drawings
											//it may not be an accurate reflection. Worst case, it isn't selected by default
			}	
			
			bufferMat.fillPoly([sketch[cnt].points], WHITE); 
		
		}
		
		//OK drawn all the lines onto the buffer. Now generate a contour object
		var contours = bufferMat.findContours();
		delete bufferMat;
		
		return { "contours": contours, "newSketches": newSketches };
	
	}
	return false;
}



function getZoomBoundaryFromCoords(x, y, screenWidth, screenHeight, photoWidth, photoHeight)
{
	//Input x,y in screen coords. screenWidth = 800 x 600 typically. photoWidth = 3000, 2000 approx
	//We want to output [ left, right, top, bottom ] in photo coords of the boundary of the photo around the x,y,
	//but the output screen dimensions are kept the same.
	
	
	var zoomScale = 0.4;				//Ratio of photo pixels to new zoomed pixels
	var zoomWidth = parseInt(photoWidth * zoomScale);	
	var zoomHeight = parseInt(photoHeight * zoomScale);
	
	var outx = (x * photoWidth) / screenWidth;
	var outy = (y * photoHeight) / screenHeight;
					
	var photoX = parseInt(outx);			//Note x/y were swapped around on purpose
	var photoY = parseInt(outy);

	var left = photoX - (zoomWidth / 2);
	if(left < 0) left = 0;			//border with the photo itself
	
	var right = left + zoomWidth;
	if(right > photoWidth) {
		right = photoWidth;
	}

	var top = photoY - (zoomHeight / 2);
	if(top < 0) top = 0;
	
	var bottom = top + zoomHeight;
	if(bottom > photoHeight) bottom = photoHeight;
	
	var retObj = {
		"left": left,
		"right": right,
		"top" : top,
		"bottom" : bottom	
	}
	return retObj;


}





function isEncoded(uri) {
  uri = uri || '';

  return uri !== decodeURIComponent(uri);
}


function convertCoords(x, y, optswidth, optsheight, imwidth, imheight, zoomedIn, zc) {
	//Inputs coords in screen-view coords (i.e. (0 - 800) x (0 - 600) typically), and convert these
	//into coordinates of the source image (e.g. (0 - 3000ish) x (0 - 2000ish).
	//If the zoomedIn option is set to true, it means we are using a zoomed in version of the photo,
	//zoomedInCoords (zc) will be [left, right, top, bottom] in coordinates from the source image.
	
	//Coords are 0,0 top left of the image.

	if(zoomedIn == false) {
		var outx = (x * imwidth) / optswidth;
		var outy = (y * imheight) / optsheight;
					
		var newX = parseInt(outx);			//Note x/y were swapped around on purpose
		var newY = parseInt(outy);
		
		var retObj = {
			"newX": newX,
			"newY": newY	
		};
	} else {
		//Zoomed in version
		var outx = (((zc.right - zc.left) * x) / optswidth) + zc.left;
		var outy = (((zc.bottom - zc.top) * y) / optsheight) + zc.top;
		
		var newX = parseInt(outx);			//Note x/y were swapped around on purpose
		var newY = parseInt(outy);
		
		var retObj = {
			"newX": newX,
			"newY": newY	
		};
	}

	return retObj;
}





function trailSlash(str)
{
	if(str.slice(-1) == "/") {
		return str;
	} else {
		var retStr = str + "/";
		return retStr;
	}

}

function capitalise(photo, medImagePath, cb) {
	//Input: photo = full photo path
	//       medImagePath = folder that contains the photos relative to 
	// 
	
	//Auto capitalise the first directory path - which means changing the default ID dir (the 
	//entry before the 
	//			test/path/photos/nhId/photo.jpg
	//becomes
	//  		test/path/photos/NHID/photo.jpg
	//But note that the root folder photos should not be affected:
	//          test/path/photos/photo.jpg
	//remains the same (and does NOT become test/path/PHOTOS/photo.jpg)
	//Returns cb(photoFileOnly, newDirPath, rootPath) 
	
	if(verbose == true) console.log("Capitalising " + photo + " at " + medImagePath);
	
	var thisCb = cb;
	
	readConfig(globalConfigFile, function(conf, err) {
		if(err) {
			//There was a problem loading the global config
			console.log("Error reading global config file:" + err);
			thisCb(photo, null, null);
		} else {
			
			if(verbose == true) console.log("Read global config OK.");
			if (typeof thisCb === 'undefined') {
				console.log("Error: callback was not defined.");
				thisCb(photo, null, null);
			
			}
			
			
			var rootPath = normalizeInclWinNetworks(medImagePath);
			if(verbose == true) console.log("Root path:" + medImagePath);
			
			
			var normalizedFile = normalizeInclWinNetworks(photo);
			normalizedFile = normalizedFile.replace(rootPath, "");
			if(normalizedFile[0] == '/') {
				normalizedFile = normalizedFile.substr(1);	//Remove any '/'s at the start
			}
			if(verbose == true) console.log("Photo file (Normalized):" + normalizedFile);		
			
			
			var pathRenaming = [];
			pathRenaming[0] = rootPath;
			var pathSections = [];
			var filenameSections = [];
			var cnt = 0;
	
			var rootPath = pathRenaming[cnt];
		
			if(verbose == true) console.log("Using rootPath:" + rootPath);
			
			//Append the normalized file to this path
			var fullFile = rootPath + normalizedFile;
			
			if(verbose == true) console.log("About to split:" + fullFile);
			
			filenameSections[cnt] = normalizedFile.split("/");
			pathSections[cnt] = fullFile.split("/");
			
			if(verbose == true) console.log("Done split");	
			
			if((filenameSections[cnt].length > 1) && (filenameSections[cnt][0]) && (filenameSections[cnt][0] != '')) {
				//We have an NHID.
				
				if(verbose == true) console.log("Have NHID");	
				
				var photoFilename = pathSections[cnt].pop();		//Remove the file itself
				
				if(verbose == true) console.log(JSON.stringify(filenameSections[cnt]));
				if(verbose == true) console.log(JSON.stringify(pathSections[cnt]));
				var oldPath = pathSections[cnt].join("/");

				if(verbose == true) console.log("oldPath: " + oldPath);


				//Make all the sections in the filename upper case
				var fnSections = filenameSections[cnt].length - 1;		//The minus 1 doesn't include the trailing filename itself
				var ptSections = pathSections[cnt].length - fnSections;		//Get the difference for the 'path only' sections
				for(var section = 0; section < fnSections; section ++) {
					pathSections[cnt][ptSections + section] = pathSections[cnt][ptSections + section].toUpperCase();
				}

				var newPath = pathSections[cnt].join("/");
				
				if(verbose == true) console.log("newPath: " + newPath);
				
				var oldDirPath = oldPath;
				var newDirPath = newPath;
				if(verbose == true) console.log("Renaming path " + oldDirPath + " to " + newDirPath);
	
				if(oldDirPath != newDirPath) {
					//Wait until photo file is in the old path fully
		
					var platform = process.platform;
					var isWin = /^win/.test(platform);
					if(isWin) {
						//Ensure the new folder is made
						fsExtra.mkdirsSync(newDirPath);
						
						
						//If this is a Windows shared drive, we want to change into the folder first.
						var winSharedDrive = false;
						if((oldDirPath[1]) && (oldDirPath[1] == ':') && (oldDirPath[0].toUpperCase() != 'C')) {
							winSharedDrive = true;
							var oldCwd = process.cwd();
							//Very Likely a Windows path. 
							//Change the drive folder.
				
							process.chdir(oldDirPath.charAt(0).toUpperCase() + ":/");
							
						}
						
						
						
						//We only need to rename the directory	
						fs.rename(oldDirPath, newDirPath, function(err) {
							if(err) {
								console.log("Error renaming path: " + err);
								
								//If the folder is already made, return the new path anyway, else return the old path
								fs.lstat(newDirPath, function(err, stats) {
									if(!err) {
										//OK the 
										if(verbose == true) console.log("Checking if folder exists status:" + stats.isDirectory());
										if(stats.isDirectory()) {
											thisCb(newDirPath + '/' + photoFilename, newDirPath, rootPath);
										} else {
											//OK, some other issue - use the old folder
											thisCb(oldDirPath + '/' + photoFilename, oldDirPath, rootPath);	
										}
									} else {
										//An error, use the old non capitalised folder
										console.log("Checking if folder exists status:" + stats.isDirectory());
										if(stats.isDirectory()) {
											thisCb(newDirPath + '/' + photoFilename, newDirPath, rootPath);
										} else {
											//OK, some other issue - use the old folder
											thisCb(oldDirPath + '/' + photoFilename, oldDirPath, rootPath);	
										}
									
									}
								
								});
								
									
								
							} else {
								
								if(verbose == true) console.log("Calling back from renaming.");										
								thisCb(newDirPath + '/' + photoFilename, newDirPath, rootPath);	
								
							}
							
							if(winSharedDrive == true) {
								//Change back to the old drive anyway
								process.chdir(oldCwd);
							}
							
						});
						
					} else {
						//This was the way we did it on unix (tested OK) - this may have preserved all old
						//content which was at the old name? But it seems like overkill. Would a
						//simple rename suffice?
					
					
						//Ensure the new folder is made
						fsExtra.mkdirsSync(newDirPath);
		
						
						//Copy all contents into the new folder (only overwrite if already there)
						fsExtra.copy(oldDirPath, newDirPath, { overwrite: true, preserveTimestamps: true }, function(err) {
							if(err) {
								console.log("Error copying older path to new path: " + err);
								
								if(verbose == true) console.log("Removed old folder cnt=" + cnt);	

								//If the folder is already made, return the new path anyway, else return the old path
								fs.lstat(newDirPath, function(err, stats) {
									if(!err) {
										//OK the 
										if(verbose == true) console.log("Checking if folder exists status:" + stats.isDirectory());
										if(stats.isDirectory()) {
											thisCb(newDirPath + '/' + photoFilename, newDirPath, rootPath);
										} else {
											//OK, some other issue - use the old folder
											thisCb(oldDirPath + '/' + photoFilename, oldDirPath, rootPath);	
										}
									} else {
										//An error, use the old non capitalised folder
										if(verbose == true) console.log("Checking if folder exists status:" + stats.isDirectory());
										if(stats.isDirectory()) {
											thisCb(newDirPath + '/' + photoFilename, newDirPath, rootPath);
										} else {
											//OK, some other issue - use the old folder
											thisCb(oldDirPath + '/' + photoFilename, oldDirPath, rootPath);	
										}
									
									}
								
								});
										
								
							} else {
				
								//Then remove the old folder and contents - if on unix.						 
								var topLevelFolder = path.join(rootPath, filenameSections[cnt][0]);
								if(verbose == true) console.log("About to remove top-level folder " + topLevelFolder);		
								
								fsExtra.removeSync(topLevelFolder);		//Removes the whole tree off the 1st folder name. Equiv to 'rm -rf'
								if(verbose == true) console.log("Removed top-level folder " + topLevelFolder);		
									
								
								//If last entry i.e. the before-backup path is returned
								thisCb(newDirPath + '/' + photoFilename, newDirPath, rootPath);	
								
							}
						});
					}	
		
				
					
				} else {
					//OK - it was the same path, but we still need to return
					thisCb(photo, null, rootPath);	
										
				
				}	 //end of 	oldDirPath == newDirPath
			}		//end of valid filename sections
		} //end of successful read config	
		
	});	//end of read config

}





  
function insertSticker(conf, areaSquarePixels, points, woundArea)
{

	

	var area = {
			"type": "sticker",
			"status": "add",
			"handDrawing": false,
			"hideMidpoints": false,
			"areaSquarePixels": areaSquarePixels,
			"areaSquareCm": Number(conf.input.sticker.physicalAreaSquareCm),
			"midPointPixels": {
				"x": 0,
				"y": 0
			},
			"points": points
		};
	
	if(!conf.output.areas) conf.output.areas = [];
	
	//Remove any existing sticker areas
	for(var cnt = 0; cnt< conf.output.areas.length; cnt++) {
		if(verbose == true) console.log("Type is " + conf.output.areas[cnt].type);
		if(conf.output.areas[cnt].type === "sticker") {
			conf.output.areas.splice(cnt);
			if(verbose == true) console.log("Removed sticker area " + cnt);
		}
	}
	
	
	conf.output.areas.push(area);
	conf.output.stickerSquarePixels = areaSquarePixels;
	if(!conf.output.fullWound.areaSquareCm) {
		conf.output.fullWound.areaSquareCm = woundArea;
	}
	
	conf = refresh.calculateDrawingArea(conf);

	
	return conf;


}  
  

function detectSticker(cv, conf, im)
{
	//Image process and try to auto-determine the sticker boundary as a circular shape
	
	 /* To test an output part way through the process:
	  
	  var imTest = new cv.Mat();	  
	  cv.cvtColor(im, imTest, cv.COLOR_GRAY2RGB, 0);		//Optional, if in grayscale
	  new jimp({
			width: imTest.cols,
			height: imTest.rows,
			data: Buffer.from(imTest.data)
		  }).write("C:/testing/partial.png");
	 	console.log("Export completed"); //TESTING 
	 */	
	 
	 //To test draw the contours with random colours
	 /* 
		var dst = cv.Mat.zeros(im.rows, im.cols, cv.CV_8UC3);
		for (let i = 0; i < contours.size(); ++i) {
			let color = new cv.Scalar(Math.round(Math.random() * 255), Math.round(Math.random() * 255),
				                      Math.round(Math.random() * 255));
			cv.drawContours(dst, contours, i, color, 1, cv.LINE_8, hierarchy, 100);
		}
	*/
	 
	
	//Loop through all the colour ranges and try to find the largest circular shape
	//that represents the sticker.
	var largestI = null;
	var largestArea = 0;
	var largestContours = {};


	for(var cnt = 0; cnt < conf.input.sticker.colourRanges.length; cnt++) {

	  var im_canny = im.clone();	
	  cv.cvtColor(im, im_canny, cv.COLOR_RGB2HSV);


	  var rng = conf.input.sticker.colourRanges[cnt];

	  //We use the range 0-180 for hue rather than 0-360

	  if(!rng.lowerHSV[3]) {
	  	rng.lowerHSV[3] = 0;
	  }
	  
	  if(!rng.upperHSV[3]) {
	  	rng.upperHSV[3] = 255;
	  }

	  if(verbose == true) console.log("Checking from range:" + JSON.stringify(rng.lowerHSV) + " to " + JSON.stringify(rng.upperHSV));

	  var im_ranged = new cv.Mat();	  
	  var low = new cv.Mat(im_canny.rows, im_canny.cols, im_canny.type(), rng.lowerHSV);
	  var high = new cv.Mat(im_canny.rows, im_canny.cols, im_canny.type(), rng.upperHSV);
	  cv.inRange(im_canny, low, high, im_ranged);
	  //Output 'im_ranged' is grayscale 8-bit image
	  

	  

	  if(verbose == true) console.log("Blur: " +JSON.stringify(rng.gaussianBlur));

	  var ksize = new cv.Size(rng.gaussianBlur[0], rng.gaussianBlur[1]);
	  cv.GaussianBlur(im_ranged, im_ranged, ksize, 0, 0, cv.BORDER_DEFAULT);


	  if(verbose == true) console.log("Edge detect from:" + JSON.stringify(rng.edgeDetectLower) + " to " + JSON.stringify(rng.edgeDetectUpper));
	  cv.Canny(im_ranged, im_ranged, rng.edgeDetectLower, rng.edgeDetectUpper, 3, false);





	  if(verbose == true) console.log("Dilation: " + rng.dilationIterations);
	  var M = cv.Mat.ones(5, 5, cv.CV_8U);
	  var anchor = new cv.Point(-1, -1);
	  // You can try different parameters
      cv.dilate(im_ranged, im_ranged, M, anchor, rng.dilationIterations, cv.BORDER_CONSTANT, cv.morphologyDefaultBorderValue());

	

	  var contours = new cv.MatVector();
	  var hierarchy = new cv.Mat();
	  cv.findContours(im_ranged, contours, hierarchy, cv.RETR_CCOMP, cv.CHAIN_APPROX_SIMPLE);

	  for(i = 0; i < contours.size(); i++) {
		

		var thisContour = contours.get(i);
		if(verbose == true) console.log("Size of thisContour before approx:" + thisContour.data32S.length);
		if(cv.contourArea(thisContour) > conf.input.sticker.minAreaSquarePixels) {
			
			var arcLength = cv.arcLength(thisContour, true);
			
			if(verbose == true) console.log("Arc length:" + arcLength);	
			
			cv.approxPolyDP(thisContour, thisContour, 0.01 * arcLength, true);		//Should be 0.01 * arcLength
			
			var cornerCount = thisContour.data32S.length/2;

			if(verbose == true) console.log("Size of thisContour after approx:" + cornerCount);	

			//See equivalent cornerCount explanation here: https://stackoverflow.com/questions/59334122/how-can-i-get-coordinates-of-points-of-contour-corners-in-opencv-js
			if((cornerCount > 5) && (cornerCount <= 20)) {
				 //Likely to be a circle between 9 - 16
				if(verbose == true) console.log("Area:" + cv.contourArea(thisContour));
				
				if(verbose == true) console.log("Corner count:" + cornerCount);         

				if(cv.contourArea(thisContour) > largestArea) {
				  if(verbose == true) console.log("Setting largest to be:" + i);         
				
				  largestArea = cv.contourArea(thisContour);
				  largestI = i;	
				  largestContours = thisContour;
				}
			}

		}  //end of 
	  }  //end of sticker contours
	}  //end of stricker colour ranges

	

	  

	if(largestI !== null) {
	   //There appears to be a circular shape in the picture
	   if(verbose == true) console.log("Sticker is " + largestArea + " pixels squared");
	   conf.output.stickerSquarePixels = largestArea;
	   
	   //Now add the contours so that the edge can be edited.
	   // Access vertex data of contours
	   
	   var points = [];

			  
	  for(var i = 0; i < largestContours.data32S.length; i+=2) {
	
			var point = {};
			point.x = largestContours.data32S[i];
			point.y = largestContours.data32S[i+1];
			
			if(verbose == true) console.log("(" + point.x + "," + point.y + ")");
			var thisPoint = [ point.x, (conf.output.scaledPhoto.height - point.y)   ];		//Was getting coords upside down
			points.push(thisPoint);	
	
	  }

		


	   if(verbose == true) console.log("Before insert sticker: " + JSON.stringify(conf, null, 4));
	   conf = insertSticker(conf, largestArea, points, 0);
	   if(verbose == true) console.log("After insert sticker: " + JSON.stringify(conf, null, 4));
	   
	} else {
		if(verbose == true) console.log("Sorry, no sticker was detected.");
		//Do not insert a sticker in here.
	
	}

	return conf;
	

}

function calculateMidpoint(config, areaId)
{
	//Finds the midpoint of a particular area
	var points = config.output.woundAreas[areaId].points;

	var xmid = 0;
	var ymid = 0;

	for(var cnt = 0; cnt< points.length; cnt++) {
		xmid += points[0];
		ymid += points[1];
	}
	
	config.output.woundAreas[areaId].midPointPixels = {
		"x": Math.round(xmid / points.length),
		"y": Math.round(ymid / points.length)
	}
	
	return config;
	
}



function getBackupFoldersCapitalised(inputPhoto, back) {
	//Input an array of input photo names,
	//Output capitalised versions of those photo file names, with the output folder correctly created if not done
	//already.
	
	//Returns cb(renamedFolder)
	
  	//Capitalise backup folders 
  	readConfig(globalConfigFile, function(globalConf, err) {
		if(err) {
			//There was a problem loading the global config
			console.log("Error reading global config file:" + err);
			back("Error reading global config file:" + err);
		} else {

			if(verbose == true) console.log("Read global config OK.");
			
			if(globalConf.backupTo && globalConf.backupTo[0]) {
				var mainMedImagePathAfterBackup = trailSlash(globalConf.backupTo[0]);	  //Relative to __dirname? or process.cwd()?
			} else {
				//Leave as-is, global
				var mainMedImagePathAfterBackup = global.mainMedImagePathAfterBackup;
			}
			
			if(verbose == true) console.log("mainMedImagePath after backup: " + mainMedImagePathAfterBackup);
			
			var photoFilename = mainMedImagePathAfterBackup + inputPhoto;
			
			//Now capitalise on the global backup folder
			capitalise(photoFilename, mainMedImagePathAfterBackup, function(photoFileOnly, newDirPath, rootPath) {
				 //Do nothing here - this is prepping the way for the correct folder name in the target backup dirs.
				if(verbose == true) console.log("Finished capitalising target folder.");
				back(null);
			});
								
		}
	});
	
	return;

}



async function processPhoto(cv, photoFile, opts, cb) {

  if(verbose == true) console.log("Reading photo file:" + photoFile);
  if(verbose == true) console.log("Reading relative photo file:" + opts.originalPath);

  var readConfigFile = photoFile.replace(".jpg", ".json");
  var relativeReadConfigFile = opts.originalPath.replace(".jpg", ".json");
  var writeConfigFile = readConfigFile;		//we still want to write to this one, even if we don't read from it

  if (fs.existsSync(readConfigFile)) {
	 //Yes, this photo already has a json file to use
	 if(verbose == true) console.log("Using existing photo-specific config file.");
  } else {
  	  //OK check if a file off from the root folder exists
  	  readConfigFile = path.resolve(path.join(__dirname, mainMedImagePath, relativeReadConfigFile));
  	  if (fs.existsSync( readConfigFile)) {
  	  		if(verbose == true) console.log("Using existing absolute photo-specific config file: " + readConfigFile);
  	  		photoFile = path.resolve(path.join(__dirname, mainMedImagePath, photoFile));		//Also make absolute
  	  		if(verbose == true) console.log("Photo file is now: " + photoFile);
  	  		writeConfigFile = readConfigFile;
  	  } else {
  	  			
			//Otherwise, use the master file by default
			if(verbose == true) console.log("Using master config file. " + masterConfigFile + " as the specific config " + readConfigFile + " was not found.");


			readConfigFile = masterConfigFile;			//Change the read config file
				
	  }
  }


	readConfig(readConfigFile, function(conf, err) {

	   if(err) {
		 console.log("Error reading config file:" + err);
		 process.exit(0);
		 //TODO: proper return value here
	   }
	   
	   
	 
	   
	   	if(verbose == true) console.log("About to read photo:" + photoFile);
	
		//Try to resize the image with imageMagick, and if so, resize the image down to something more manageable, e.g. 1500 pixels rather than 4000
		//TODO: warning, use a parameterized version of exec here
		var resizedPhotoFile = photoFile.replace(conf.output.scaledPhoto.nameConvention.replaceText, 
													".resized.jpg");
	    var execRun = "magick " + photoFile + " -resize 1200x1200 " + resizedPhotoFile;
	    var child = spawn("magick", [photoFile, "-resize", "1200x1200", resizedPhotoFile]);
	    
	    //exec(execRun, function(error, stdout, stderr){			//The exec works but is a security risk
		child.on('error', (error) => {
			console.log("Error running imagemagick:" + error + ". Will continue with the slower NodeJS method, instead. Please install ImageMagick (https://imagemagick.org) for a faster response time.");
		});
		child.on('close', (data) => {
			 if(verbose == true) console.log("ImageMagick closing code: " + data);
	
			 if (fs.existsSync(resizedPhotoFile)) {
			 	var readingFile = resizedPhotoFile;
			 	 if(verbose == true) console.log("Reading smaller version of file " +  resizedPhotoFile);
			 } else {
			 	var readingFile = photoFile;
			 	 if(verbose == true) console.log("Reading larger version of file " +  photoFile);
			 }
 			
 	
		
			// load local image file with jimp. It supports jpg, png, bmp, tiff and gif:
	  	    jimp.read(readingFile, function(err, jimpIm) {
	  	   
			
			
			  if (err) throw new Error("Error reading photo:" + photoFile + " Error:" + err);

			  if(verbose == true) console.log("Finished reading photo:" + photoFile + " About to convert into in-memory version...");

	 		  // `jimpImage.bitmap` property has the decoded ImageData that we can use to create a cv:Mat
	 	      var incomingIm = cv.matFromImageData(jimpIm.bitmap);
	 	      
	 	       if(verbose == true) console.log("Converted photo to in-memory version" + photoFile);
	 	      
	 	      if(incomingIm.rows > incomingIm.cols) {
	 	      	//Rotate a vertical photo
	 	      	var im = new cv.Mat();
	 	      	//See https://stackoverflow.com/questions/52963949/image-auto-cropping-when-rotate-in-opencv-js
	 	      	//3 options: cv.ROTATE_90_CLOCKWISE   cv.ROTATE_180   cv.ROTATE_90_COUNTERCLOCKWISE
	 	      	if(verbose == true) console.log("Vertical, so starting rotation of " + photoFile);
	 	      	cv.rotate(incomingIm, im, cv.ROTATE_90_CLOCKWISE);
				if(verbose == true) console.log("Finished vertical rotation of " + photoFile);
	 	      

	 	      } else {
	 	      	var im = incomingIm.clone();
	 	      }

			  conf.input.thisPhotoPath = photoFile;			//Set the current photo into the config
			  conf.input.scriptRootPath = __dirname + '/';

			  var width = im.cols;
			  var height = im.rows;
			  conf.input.photoSize.width = width;
			  conf.input.photoSize.height = height;

				 

			  if (width < 1 || height < 1) throw new Error("Sorry, the image " + photoFile + " needs to have a valid size. Width =" + width + " Height=" + height);

			  var outputPhoto = new cv.Mat(height, width, 24);

		  	    
		
			  outputPhoto = im.clone();
		 
		 	  if(verbose == true) console.log("Finished cloning of " + photoFile);
	  		  
			  
	  
			  //Rescale original photo for quick web view of the original. No other drawings on this.
			  var webPhotoFile = photoFile.replace(conf.output.scaledPhoto.nameConvention.replaceText, 
													conf.output.scaledPhoto.nameConvention.withWebViewText);
			  
			  if(verbose == true) console.log('Resizing and creating web photo...' + webPhotoFile);
			  
		  
			  //Output a resized version to disk
			  new jimp({
				width: outputPhoto.cols,
				height: outputPhoto.rows,
				data: Buffer.from(outputPhoto.data)
			  })
			  .resize(conf.output.scaledPhoto.width, conf.output.scaledPhoto.height) // resize
			  .quality(95) // set JPEG quality
			  .write(webPhotoFile); 
			  
			  //resize the image into a local buffer, called webPhoto	  
			 var resizedJimpIm = new jimp({
				width: outputPhoto.cols,
				height: outputPhoto.rows,
				data: Buffer.from(outputPhoto.data)
			  })
			  .resize(conf.output.scaledPhoto.width, conf.output.scaledPhoto.height);
			 var webPhoto = cv.matFromImageData(resizedJimpIm.bitmap);

			if(verbose == true) console.log("Finished resizing and creating web photo " + webPhotoFile + ". About to detect sticker.");

			  
			   //Auto-detect the sticker off the smaller photo
			  conf = detectSticker(global.cv, conf, webPhoto);

			  if(verbose == true) console.log('Detected sticker');
			  
			  //Now we're done with this image in memory
			  delete webPhoto;

			 
			  
			  refresh.refreshThumbnail(cv, photoFile, webPhotoFile, conf, function(err, thumbnailPhotoFile) {
						
						
				  if(err) {
				  	   //And create a thumbnail anyway
					  var thumbnailPhotoFile = photoFile.replace(conf.output.scaledPhoto.nameConvention.replaceText, 
															conf.output.scaledPhoto.nameConvention.withThumbnailText);

					  if(verbose == true) console.log('Saving thumbnail photo...' + thumbnailPhotoFile);
					  new jimp({
						width: webPhoto.cols,
						height: webPhoto.rows,
						data: Buffer.from(webPhoto.data)
					  })
					  .resize(conf.output.scaledPhoto.thumbnailWidth, conf.output.scaledPhoto.thumbnailHeight) // resize
					  .quality(95) // set JPEG quality
					  .write(thumbnailPhotoFile);
					  
					  
					  delete thumbnailPhoto;
					  if(verbose == true) console.log('Done');
				  }


				  if(verbose == true) console.log('All images saved.');
	  
	  
	  
	  

				  //Now write back the config to the correct path
				  writeConfig(writeConfigFile, conf, function(err) {
					  if(err) {
							throw new Error("Cannot write photo config file " + writeConfigFile + "  err:" + err);
							//TODO: do a proper callback here
					  } else {
	  
						  if(verbose == true) console.log("All finished! Written:" + writeConfigFile);

							
						   getBackupFoldersCapitalised(opts.originalPath, function(err) {
	  
							   //Back up all the created/modified files to any secondary directories the user has specified in their MedImage server config
							   var backupStr = "";      //Old way: kept duplicates: "backupFiles:"  + resolve(webPhotoFile) + ";" + resolve(thumbnailPhotoFile) + ";" + resolve(writeConfigFile) + "\n";
							   if(verbose == true) console.log(backupStr);
							   //Without the main photoFile (that gets backed up by default anyway): + resolve(photoFile) + ";"


							   //Check if we want to display the sticker version instead of the default wound version
							   var startWith = "wound";
							   if((conf) && (conf.output)) {
									if(!conf.output.stickerSquarePixels) {		//Either 0 or not existing
										startWith = "sticker";
									}
							   } else {
									startWith = "sticker";
							   }
	   
							   if(opts.startWith == "sticker") {
									startWith = "sticker";
							   }



							  if(conf.output.fullWound) {
								  var outputArea = conf.output.fullWound.areaSquareCm;

							  } else {
								  var outputArea = "[Unknown Area]";
							  }
							  
							  if((conf) && (conf.output) && (conf.output.fullWound) && (conf.output.fullWound.underAreaSquareCm)) {
									
									var woundArea = conf.output.fullWound.woundAreaSquareCm;
									var underminingArea = conf.output.fullWound.underAreaSquareCm;
									var underminingAreaStr = ".<span style=\"color: #888;\"> External <i class=\"fa fa-pie-chart\"></i>  <span id=\"total-area\">" + woundArea + "</span> cm<sup>2</sup>.  Undermining <i class=\"fa fa-pie-chart\"></i>  <span id=\"total-area\">" + underminingArea + "</span> cm<sup>2</sup></span>";
							   } else {
									var underminingAreaStr = "";
							   }
							  
	  
							  if(conf.output.areas) {
								  var areas = encodeURIComponent(JSON.stringify(conf.output.areas));
							  } else {
								 var areas = "[]";
							  }


							  var returnLink = "";
							  if(opts.ret) {
								 returnLink = opts.ret;
							  }

							  var sessionId = "";
							  if(opts.sessionId) {
								 sessionId = opts.sessionId;
							  }
	  
	  							
							  var webRelativePhotoFile = opts.originalPath.replace(conf.output.scaledPhoto.nameConvention.replaceText, 
															conf.output.scaledPhoto.nameConvention.withWebViewText);
	  
	  
	   						   //Display wound type classification
							   var startWoundType = "Unclassified";
							   if((conf) && (conf.output)) {
									if(conf.output.woundType) {		
										startWoundType = conf.output.woundType;
									} 
							   }
	  							
							  var returnVal = backupStr + "returnParams:?CUSTOMAREA=" + outputArea + "&CUSTOMIMAGE=" + opts.originalPath + "&CUSTOMWOUNDIMAGE=" + webRelativePhotoFile + "&AREAS=" + areas + "&UNDERMININGAREA=" + underminingAreaStr + "&CUSTOMSTARTWITH=" + startWith + "&CUSTOMWOUNDTYPE=" + startWoundType + "&RANDIMAGE=" + Date.now() + "&RETLINK=" + returnLink + "&SESSIONID=" + sessionId;
							  
							  cb(null, returnVal);
						  });		//End of getBackupFoldersCapitalised
						}
				   });	//End of writeConfig
					
			  });			//End of refresh thumbnail


			  //removed cb() here?
			});  //end of cv.readImage

		}); //End of imagemagick exec()
		
	   
	   
	});	//End of read config
	

}







async function onRuntimeInitialized() {
	  // load local image file with jimp. It supports jpg, png, bmp, tiff and gif:
  
	global.cvIsReady = true;
	if(verbose == true) console.log("CV runtime wound-size.js is ready!");

}


//Note: must be capital 'M' module
global.Module = {
  onRuntimeInitialized
}		//End of module


//Note: must be lower case 'm' module here.
module.exports = {
	medImage : function(argv, callback) {
		//Note: all file reading and writing should be in this function (for some reason - I haven't yet determined why but that is the practical reality).

	

 		var callMedImageAgain = module.exports.medImage;
	
		if(!global.cvIsReady) {
			setTimeout(function() {
				var resp = "CV image processing runtime is not yet ready. We are trying again in 3 seconds.";
				
				
				//Always return { err, stdout, stderr }. Include an error in the first param if necessary.
				if(verbose == true) console.log(resp);
			
				callMedImageAgain(argv, callback);
				return;
			}, 3000);
			return;
		}
        
		//Starting from here.
		var photoFile = "";


		if(argv[0]) {
		 
		   var opts = {};
		   
		   if(isEncoded(argv[0])) {
		  	 //Then this is from a web form.
		  	 var opts = queryString.parse(decodeURIComponent(argv[0]));
		  	 webForm = true;
		  	 //opts.photo = queryString.parse(decodeURIComponent(argv[0]));

		  	 photoFile = normalizeInclWinNetworks(__dirname + "/" + mainMedImagePath + opts.photo);		
		  	 
		  	  if(verbose == true) if(verbose == true)console.log("Web operation photo file path converted to:" + photoFile);
		  	 opts.photo = photoFile;		//Trying this.
		  
		  } else {
		  
		  	//Read the command line argument for the photo to process
		  	photoFile = normalizeInclWinNetworks(argv[0]);		//This should be relative to the script folder
		  	
		  	if(verbose == true) console.log("Command line operation photo file path converted to:" + photoFile);
		  	
		  	opts.photo = photoFile;		//Trying this.
		  	webForm = false;
		  }
		  
		  //photoFile should be e.g. C:/MedImage/photos/test/photo-datetime.jpg
		 

		  
		  opts.displayPath = opts.photo;
		  
		  //originalPath is always the short version of the photo file as it was entered eg. TEST/thisphoto.jpg or Test/thisphoto.jpg
		  var srcFolder = normalizeInclWinNetworks(__dirname + "/" + mainMedImagePath);
		  opts.srcFolder = srcFolder;
		  if(verbose == true)  console.log("Source folder path: " + opts.srcFolder);
		  if(verbose == true)  console.log("Source photo: " + opts.photo);
		  opts.originalPath = opts.photo.replace(srcFolder, "");
		  if(verbose == true) console.log("Original path: " + opts.originalPath);
		  
		 
		 
		  readConfig(masterConfigFile, function(masterConf, err) {

			   if(err) {
			   	 console.log("Error reading config file:" + err);
			  	 process.exit(0);
			   }
			   
			   if(masterConf.output && masterConf.output.autoCapitaliseID && masterConf.output.autoCapitaliseID == true) { 
					 			 
					 var medImagePathBeforeBackup = normalizeInclWinNetworks(__dirname + "/" + mainMedImagePath);			 
					 
					 
					 			 
					 capitalise(photoFile, medImagePathBeforeBackup, function(photoFileOnly, newDirPath, rootPath) {
					 		
					 		//Finished capitalise
					 		if(verbose == true) console.log("Finished capitalise. Output photo:" + photoFileOnly + "  Output newDir:" + newDirPath +  "  rootPath:" + rootPath);
					 		
					 		
					 		
					 		
					 		opts.displayPath = photoFileOnly.replace(rootPath, "");	
					 		
					 		if(verbose == true) console.log("displayPath:" + opts.displayPath);
					 		
					 		
					 		var platform = process.platform;
							var isWin = /^win/.test(platform);
								
					 		
					 		
					 		//Rename the main one - photoFile
							if(webForm == true) {
								//We want the long version, but the new version
								if(newDirPath) {
									if(photoFileOnly.search(newDirPath) >= 0) {
										//Already exists in there
										photoFile = photoFileOnly;
									} else {
										photoFile = newDirPath + "/" + photoFileOnly;
									}
									
									
									if(isWin) {
										photoFile = photoFile.replace(rootPath, "");		//Not sure this replace is needed any more?						
										opts.photo = photoFileOnly.replace(rootPath, "");		//This should only be the display path.
									} else {
										opts.photo = photoFileOnly;
									}
								} else {
									//Which would be a complete photo path in this case anyway - leave as-is
								}
								if(verbose == true) console.log("Web operation photo file path changed to:" + photoFile);   //if(verbose == true) 
							} else {
								//We want the short version 
								if(isWin) {
									//On win we only want to update the opts.photo, but not the main folder passed in
									photoFileOnly = photoFileOnly.replace(rootPath, "");
								} else {
									//On linux, we only want the short version
									photoFile = photoFileOnly;	
								}
								opts.photo = photoFileOnly;		//This should only be the display path.	
								
								if(verbose == true) console.log("Command line photo file path changed to:" + photoFile);   //if(verbose == true) 
							}
							
													
							//Actually process the photos
							processPhoto(global.cv, photoFile, opts, function(err, output) {
								if(!output) {
									var resp = "There was an error processing the wound files.";
									output =  "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));  
								}

								var ret = {};
								ret.err = "";
								ret.stdout = output;
								ret.stderr = "";
								
								//Always return { err, stdout, stderr }. Include an error in the first param if necessary.
								callback(null, ret);	
								return;	
							
							});
							
							
					 });		//End of capitalise	
				} else {
						
					//Actually process the photos
					processPhoto(global.cv, photoFile, opts, function(err, output) {
						if(!output) {
							var resp = "There was an error processing the wound files.";
							output =  "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));  
						}

						var ret = {};
						ret.err = "";
						ret.stdout = output;
						ret.stderr = "";
						
						//Always return { err, stdout, stderr }. Include an error in the first param if necessary.
						callback(null, ret);	
						return;	
					
					});
					
					
		  
		  		}
		  	});  //End of read master config


		} else {

			console.log("Usage: node wound-size.js photo/path/file.jpg");
		}

	} //End of medimage()
}


// Load 'opencv.js' assigning the value to the global variable 'cv'
if(typeof global.cv === 'undefined'){
	global.cv = require('./opencv.js');	
}

