/*

MedImage Wound Analysis Add-on - Hue grabber
==============================

Developed by (C) AtomJump Ltd. 10 Aug 2017. This is a commercial add-on to MedImage, and is licensed
by AtomJump on a per-server basis.

This script accepts an input .json file that defines how it will be run.

The json file includes a photo filename, and an x,y coordinate within that photo. 
It also gives the choice of
  - low end of HSV colour range, or upper end, or use an auto range around the selected colour
  - wound colours or sticker colours
  - master or disciple json
  
The script will read the photo file, grab the RGB of that coordinate, convert to HSV and write 
the value into a new colour range object in either the config/master.json, or the photo's json file.

e.g. but now as parameters in a get request 
{
    "photo": "/var/www/html/medimageservtest/photos/yo/9-Aug-2017-15-17-45.jpg",
    "x": 1386,
    "y": 1955,
    "lowRange": true,
    "type": "wound",
    "master": false,
    "keep": true
}


Note:  x is from the right side of the photo (= 0) moving left and increasing.
       y is from the top side of the photo (=0) move down and increasing.


To run this from the command line you should now use e.g.


node hue-due.js photo%3D27-Nov-2023-10-41-42.jpg%26x%3D345%26y%3D678%26lowRange%3Dtrue%26type%3Dwound%26master%3Dtrue%26width%3D1200%26height%3D800

which is:
photo=27-Nov-2023-10-41-42.jpg&x=345&y=678&lowRange=true&type=wound&master=true&width=1200&height=800
decoded.

*/


var jimp = require('jimp');

var fs = require('fs');
var fsExtra = require('fs-extra');
var upath = require('upath');
var path = require('path');
var queryString = require('querystring');
var exec = require('child_process').exec;
var resolve = require('path').resolve;

var verbose = false;		//usually false

var AUTO_HUE_RANGE = 3;				//Range in hue (H in an HSV colour space)
var AUTO_SATURATION_RANGE = 50;		//Range in saturation (S in an HSV colour space)
var AUTO_VALUE_RANGE = 50;			//Range in brightness (V in an HSV colour space)



var masterConfigFile = __dirname + '/config/master.json';
var globalConfigFile = __dirname + '/../../config.json';
var relImagePath = "../../photos";
var mainMedImagePath = path.relative(process.cwd(), __dirname + '/' + relImagePath);

var cvHueIsReady = false;




//Include shared funcs
var normalizeInclWinNetworks = require(__dirname + '/common/libfuncs.js').normalizeInclWinNetworks;
var readConfig = require(__dirname + '/common/libfuncs.js').readConfig;
var writeConfig = require(__dirname + '/common/libfuncs.js').writeConfig;




function checkRangeExist(colourRanges, newRange) {
	//returns false if the newRange doesn't exist, true if it does already
	var newRangeStr = JSON.stringify(newRange);
	for(var cnt=0; cnt< colourRanges.length; cnt++) {
		if(JSON.stringify(colourRanges[cnt]) === newRangeStr) {
			return true;
		}	
	}
	
	return false;
}


function determineConfigUsed(opts, globConf) {
		//Write this array into the range calculation in the config file
		var readConfigFile = opts.photo.replace(".jpg", ".json");
		
		if ((opts.master == "false")||(opts.master == false)) {
				 //Yes, this photo already has a json file to use
				 if(verbose == true) console.log("Using existing photo-specific config file.");
		} else {
			//Otherwise, use the master file by default
			if(verbose == true) console.log("Using master config file.");
			readConfigFile = masterConfigFile;
			
			//Note: possible improvement here - we can visibly affect the current photo also with this new colour.
		}


	

		//With this global config, determine whether we should use a remote version
		if(globConf.backupTo && globConf.backupTo[0]) {
			//By default use the destination version, since that is shared. It sits in the home folder of the shared drive folder						
			
			var sharedMasterConfigFile = globConf.backupTo[0] + "/master.json"; 
			try {
			  if (fs.existsSync(sharedMasterConfigFile)) {
				//master file exists
				//And copy shared master down to local master
				try {
				  fsExtra.copySync(sharedMasterConfigFile, masterConfigFile);
				  if(verbose == true) console.log('Copied shared master ' + sharedMasterConfigFile + ' to local master ' + masterConfigFile + ', success!');
				} catch (err) {
				  console.error(err);
				}
				
				
				
			  } else {
				//doesn't exist, so local master will be backed up, also
				try {
				  fsExtra.copySync(masterConfigFile, sharedMasterConfigFile);
				  if(verbose == true) console.log('Shared local master config, success!');
				} catch (err) {
				  console.error(err);
				}
			  }
			  
			  //Either way we need to read the shared version herre
			  if ((opts.master == "false")||(opts.master == false)) {
			  		//Ignore variations here
			  } else {
			  	readConfigFile = sharedMasterConfigFile;
			  }
			  
			} catch(err) {
			  console.error(err);
			}
			
			
			
		} else {
			//Otherwise use the local version
		}

		if(verbose == true) console.log("Using config file: " + readConfigFile);
		return readConfigFile;
}



function updateConfig(readConfigFile, conf, globConf, opts, origPhoto, pixel) {
		//Start of reading file config file
		 var oldNonColourFactors = null;
	   
	   
	   
	   if(opts.type == "wound") {
			var colourRanges = conf.input.wound.colourRanges;
		} else {
			var colourRanges = conf.input.sticker.colourRanges;
		}

		if(verbose == true) console.log("Colour ranges from config:" + JSON.stringify(colourRanges));
		
		if(colourRanges) {

			   if(opts.keep == "false") {
					//We want to remove what is already there
					if(colourRanges[0]) {
						oldNonColourFactors = {
							"dilationIterations": colourRanges[0].dilationIterations,
							"gaussianBlur": colourRanges[0].gaussianBlur,
							"edgeDetectLower": colourRanges[0].edgeDetectLower,
							"edgeDetectUpper": colourRanges[0].edgeDetectUpper
						};
					}
					colourRanges.length = 0;
			   }
	   
			   if(opts.lowRange == "true") {
	   
					//Create a new entry in colour ranges
					var len = colourRanges.length;
					if(colourRanges[0]) {
						var newRange = {
							"lowerHSV": pixel,
							"upperHSV": pixel,
							"dilationIterations": colourRanges[0].dilationIterations,
							"gaussianBlur": colourRanges[0].gaussianBlur,
							"edgeDetectLower": colourRanges[0].edgeDetectLower,
							"edgeDetectUpper": colourRanges[0].edgeDetectUpper,
							"comment": "Created by hue-due.js. Run again for upper."		
						};
				
				
					} else {
						//Keep the old dilation factors etc.
						if(oldNonColourFactors) {
							var newRange = {
								"lowerHSV": pixel,
								"upperHSV": pixel,
								"dilationIterations": oldNonColourFactors.dilationIterations,
								"gaussianBlur": oldNonColourFactors.gaussianBlur,
								"edgeDetectLower": oldNonColourFactors.edgeDetectLower,
								"edgeDetectUpper": oldNonColourFactors.edgeDetectUpper,
								"comment": "Created by hue-due.js. Run again for upper."		
							};
						} else {
							//Defaults
							var newRange = {
								"lowerHSV": pixel,
								"upperHSV": pixel,
								"dilationIterations": 4,
								"gaussianBlur": [7,7],
								"edgeDetectLower": 0,
								"edgeDetectUpper": 50,
								"comment": "Created by hue-due.js. Run again for upper."		
							};
				
						}
			
					}
			
					if(checkRangeExist(colourRanges, newRange) == false) {
							colourRanges[len] = newRange;
					}
			
			   } 
	   
	   
			   if(opts.lowRange == "false") {
					//Upper range, so set the last colourRange element with this upper value
					var len = colourRanges.length;
			
					var oldHSV = colourRanges[len - 1].lowerHSV;
					var newHSV = pixel;
			
					var lowerHSV = [];
					var upperHSV = [];
			
					//OK, get the lowest and the highest of the individual hsv components, and store them in that order
					for(dim = 0; dim < 3; dim++) {
						if(oldHSV[dim] < newHSV[dim]) {
							lowerHSV[dim] = oldHSV[dim];
							upperHSV[dim] = newHSV[dim];
						} else {
							lowerHSV[dim] = newHSV[dim];
							upperHSV[dim] = oldHSV[dim];
						}
					}
			
			
					colourRanges[len - 1].lowerHSV = lowerHSV;
					colourRanges[len - 1].upperHSV = upperHSV;
					colourRanges[len - 1].comment = "Created by hue-due.js";
	   
			   }
	   
			   //Or dark to light
			   if(opts.lowRange == "darkToLight") {
					//So, take this colour, get a slight delta on the hue up and down, and get a wide variation
					//of colour range from dark to light in terms of brightness and saturation
			
					//Pixel is in the correct array e.g. [104,113,170] which is Hue,Saturation,Value (0-179, 0-255, 0-255) 
					//Weird issue - it seems like 250 is the max, not 255.
					var lowH = pixel[0] - AUTO_HUE_RANGE;
					if(lowH < 0) lowH = 180 + lowH;		//wrap around
					var lowS = pixel[1] - AUTO_SATURATION_RANGE;
					if(lowS < 0) lowS = 0;
					var lowV = pixel[2] - AUTO_VALUE_RANGE;
					if(lowV < 0) lowV = 0;			   		
					var lowHSV = [lowH, lowS, lowV];		//Get full range array
			
					var highH = pixel[0] + AUTO_HUE_RANGE;
					if(highH > 180) highH = highH - 180;		//wrap around
					var highS = pixel[1] + AUTO_SATURATION_RANGE;
					if(highS > 250) highS = 250;
					var highV = pixel[2] + AUTO_VALUE_RANGE;
					if(highV > 250) highV = 250;			   		
					var highHSV = [highH, highS, highV];		//Get full range array
			
			
			
			
			
					var len = colourRanges.length;
					if(colourRanges[0]) {
						var newRange = {
							"lowerHSV": lowHSV,
							"upperHSV": highHSV,
							"dilationIterations": colourRanges[0].dilationIterations,
							"gaussianBlur": colourRanges[0].gaussianBlur,
							"edgeDetectLower": colourRanges[0].edgeDetectLower,
							"edgeDetectUpper": colourRanges[0].edgeDetectUpper,
							"comment": "Created by hue-due.js. Run again for upper."		
						};
					} else {
						//Keep the old dilation factors etc.
						if(oldNonColourFactors) {
							var newRange = {
								"lowerHSV": lowHSV,
								"upperHSV": highHSV,
								"dilationIterations": oldNonColourFactors.dilationIterations,
								"gaussianBlur": oldNonColourFactors.gaussianBlur,
								"edgeDetectLower": oldNonColourFactors.edgeDetectLower,
								"edgeDetectUpper": oldNonColourFactors.edgeDetectUpper,
								"comment": "Created by hue-due.js. Run again for upper."		
							};
						} else {
							//Defaults
							var newRange = {
								"lowerHSV": lowHSV,
								"upperHSV": highHSV,
								"dilationIterations": 4,
								"gaussianBlur": [7,7],
								"edgeDetectLower": 0,
								"edgeDetectUpper": 50,
								"comment": "Created by hue-due.js. Run again for upper."		
							};
				
						}
			
					}
			
					if(checkRangeExist(colourRanges, newRange) == false) {
							colourRanges[len] = newRange;
					}
			   }
	   
	   
	   
	   
	   
	   
	   
			 
		} else {
			//No colour range
			console.log("Warning: no colour range in config file. Perhaps you should try a different type e.g. wound/sticker?");
			return false;
		}
					 
	   //Don't return here, wait for the output
	   return conf;
}









  


async function onRuntimeInitialized() {
	  // load local image file with jimp. It supports jpg, png, bmp, tiff and gif:
  
	global.cvIsReady = true;
	console.log("CV runtime is ready!");

}


//Note: must be capital 'M' module
Module = {
  onRuntimeInitialized
}		//End of module


//Note: must be lower case 'm' module here.
module.exports = {
	medImage : function(argv, callback) {
			//Note: all file reading and writing should be in this function (for some reason - I haven't yet determined why but that is the practical reality).

	
			var callMedImageAgain = module.exports.medImage;
	
			if(!global.cvIsReady) {
				setTimeout(function() {
					var resp = "CV image processing runtime in hue-due.js is not yet ready. We are trying again in 3 seconds.";
					
					
					//Always return { err, stdout, stderr }. Include an error in the first param if necessary.
					console.log(resp);
				
					callMedImageAgain(argv, callback);
					return;
				}, 3000);
				return;
			}
	

            //Your code in here, which sets 'resp'. The output is up to you.
            
           var opts = queryString.parse(decodeURIComponent(argv));
		   var origPhoto = opts.photo;
		   opts.photo = __dirname + "/" + relImagePath + "/" + opts.photo;		
		
		
		
			//Read in the global config to see if we should be using a shared drive version of the config, and get some constants
			var globConf = global.globalConfig;	//Get the parent MedImage Server config
				
			var readConfigFile = determineConfigUsed(opts, globConf);
            
  		
			readConfig(readConfigFile, function(conf, err) {

	  
			  	 if(err) {
				 	
				 	var resp = {
						"completed": false,
						"error": "Error reading config file"	   	 
					} 
					output = "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));
					
					var ret = {};
					ret.err = "";
					ret.stdout = output;
					ret.stderr = "";
					callback(null, ret);
					return;
					
			   	 } else {
			   	 		
			   	 	var webPhotoFile = opts.photo.replace(conf.output.scaledPhoto.nameConvention.replaceText, 
									conf.output.scaledPhoto.nameConvention.withWebViewText);
			   	 	
			   	 	//Now read the photo file in
					jimp.read(webPhotoFile, function(err, jimpIm) {

							if (err) {							
								var resp = {
									"completed": false,
									"error": "Error reading photo file:" + err	   	 
								} 
								output = "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));
								
								var ret = {};
								ret.err = "";
								ret.stdout = output;
								ret.stderr = "";
								callback(null, ret);
								return;
							}
							
						
							 // `jimpImage.bitmap` property has the decoded ImageData that we can use to create a cv:Mat
							 var im = global.cv.matFromImageData(jimpIm.bitmap);
							
							
							if (im.rows < 1 || im.cols < 1) {
								var resp = {
									"completed": false,
									"error": "Error: The image has no size, or does not appear valid. Parameters input=\n" + JSON.stringify(opts, null, 2)   	 
								} 
								output = "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));
								
								var ret = {};
								ret.err = "";
								ret.stdout = output;
								ret.stderr = "";
								callback(null, ret);
								return;
								
							}
							
							
						 	if(opts.x) {
								//We need to scale these, which are between (0 - opts.width) and (0 - opts.height) usually 800x600
								//back to the scale of the web view image (given by im.width and im.height, should be 1200x800) and override the x and y. Also ensure correct orientation.
								

								var imwidth = parseInt(im.cols);		
								if(verbose == true) console.log("imwidth:" + imwidth);		
								var optswidth = parseInt(opts.width);
								var imheight = parseInt(im.rows);
								var optsheight = parseInt(opts.height);
								var outx = (opts.x * imwidth) / optswidth;
								var outy = (opts.y * imheight) / optsheight;
								
								opts.x = parseInt(outx);			
								opts.y = parseInt(outy);
								
								if(verbose == true) console.log("Pixel coords in image space = " + opts.x + "," + opts.y); 
								
							}
							
							
							
							global.cv.cvtColor(im, im, global.cv.COLOR_RGB2HSV);
							
							var pixel = im.ucharPtr(parseInt(opts.y), parseInt(opts.x));		//row, col
							
							//  var H = pixel[0];   //In 0 - 180
							//  var S = pixel[1];
							//  var V = pixel[2];
							//  var A = pixel[3];
							
							//Pixel is now the correct array e.g. [104,113,170] which is Hue,Saturation,Value (0-179, 0-255, 0-255)
							if(verbose == true) console.log('One pixel:' + JSON.stringify(pixel)); 
							
							
							conf = updateConfig(readConfigFile, conf, globConf, opts, origPhoto, pixel);					
		
								
							var sharedMasterConfigFile = globConf.backupTo[0] + "/master.json"; 
							
							
							  //And write out the config file again
							  writeConfig(readConfigFile, conf, function(err) {
									if(err) {
										if(verbose == true) console.log("Error writing json file!" + err);
										var resp = {
											"completed": false,
											"error": "Error writing .json file:" + err	   	 
										} 
										output = "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));
										
										var ret = {};
										ret.err = "";
										ret.stdout = output;
										ret.stderr = "";
										callback(null, ret);
									} else {
										//Wrote config
										if(verbose == true) console.log("wrote config!");
									
										//And write it out to the network version if that exists. Important to do before
										//running show-adjust-colours.js below, which will use the network version.
										if(globConf.backupTo && globConf.backupTo[0]) {
											//By default use the destination version, since that is shared. It sits in the home folder of the shared drive folder						
							
											var sharedFile = globConf.backupTo[0] + '/' + origPhoto.replace(".jpg", ".json");
											try {
												if(verbose == true) console.log("Saving config to network drive: " + sharedFile);
												fs.writeFileSync(sharedFile, JSON.stringify(conf, null, 6));
											} catch (err) {
											   console.error(err);
											}
										}
										
										
										
										if(verbose == true) console.log(".json file updated successfully with new colour range!");
							
										//Run a new process to display the new page			
										var run = 'node ' + __dirname + '/show-adjust-colours.js ' + argv;
										exec(run, function(error, stdout, stderr){
											if(error) {
												console.log("Error running show-adjust-colours:" + error);
												output = "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));
												callback(err, ret);
												
											} else {
												//Success - create a backup
					  							 if(readConfigFile == sharedMasterConfigFile) {
													//Want to do the backup of this file ourselves - because it isn't in the normal
													//folders. 
													//Copy global master down to local master
													try {
													  fsExtra.copySync(sharedMasterConfigFile, masterConfigFile);
													  if(verbose == true) console.log('Backing up global master ' + sharedMasterConfigFile  + ' to local ' + masterConfigFile + ' success!');
													} catch (err) {
													  console.error(err);
													}
													//No need for a backupFiles signal to the main parent MedImage
												 } else {
													if(readConfigFile == masterConfigFile) {
														//Save it up as the global master
														try {
														  fsExtra.copySync(masterConfigFile, sharedMasterConfigFile);
														  if(verbose == true) console.log('Backing up master ' + masterConfigFile  + '  to global master ' + sharedMasterConfigFile + ' success!');
														} catch (err) {
														  console.error(err);
														}
													} else {
													
														var photoFile = normalizeInclWinNetworks(resolve(readConfigFile)).toString();
													
														console.log("backupFiles:" + photoFile);  	//Must be relative
													}
												
													
												}
										
												var output = stdout;
												
												
												if(!output) {
													var resp = "";
													output =  "returnParams:?CUSTOMJSON=" + encodeURIComponent(JSON.stringify(resp));  
												}

												var ret = {};
												ret.err = "";
												ret.stdout = output;
												ret.stderr = "";
												
												//Always return { err, stdout, stderr }. Include an error in the first param if necessary.
												callback(null, ret);	
												return;	
										
											}		//End of exec valid statement
										});		//end of exec
							
								
								
								

									}		//End of successful write of config
							   });		//End of writing config file
							
							
							
							
							
							
							
					
							
							
					});	//End of jimp read of photo
			   	 
			   	 }		//End of no error reading config
				
			});  //End of read config

	}	//End of MedImage func
}


// Load 'opencv.js' assigning the value to the global variable 'cv'
if(typeof global.cv === 'undefined'){
	global.cv = require('./opencv.js');	
}

