/*

	Common library file, used by (at least) fine-tune.js and wound-size.js
	
	If you are using OpenCv calls (refreshThumbnail()), you must first include the following code in your module:
	
	async function onRuntimeInitialized() {
		  // load local image file with jimp. It supports jpg, png, bmp, tiff and gif:
	  
		global.cvIsReady = true;
		console.log("CV runtime your-script.js is ready!");		//Change your-script

	}


	//Note: must be capital 'M' module
	global.Module = {
	  onRuntimeInitialized
	}		//End of module

	
	// Load 'opencv.js' assigning the value to the global variable 'cv'
	if(typeof global.cv === 'undefined'){
		global.cv = require('./opencv.js');	
	}

*/


var jimp = require('jimp');
var fs = require('fs');
var upath = require('upath');
var queryString = require('querystring');
var path = require('path');
var area = require('area-polygon');
var resolve = require('path').resolve;
var polygon = require('@turf/helpers').polygon;
var intersect = require('@turf/intersect').default;

var verbose = false;
var globalId = "";


var masterConfigFile = __dirname + '/../config/master.json';
var mainMedImagePath = __dirname + "/../../../photos";   //If run from two dirs in: "../../photos";
var globalConfigFile = __dirname + '/../../../config.json';






module.exports.calculateDrawingArea = function(config)
{
	
	var totalWoundSquarePixels = 0;
	var totalWoundSquareCm = 0;
	
	var totalUnderSquarePixels = 0;
	var totalUnderSquareCm = 0;
	
	var totalStickerSquarePixels = 0;


	for(var cnt = 0; cnt< config.output.areas.length; cnt++) {


		if(config.output.areas[cnt].type == "sticker") {
			if(verbose == true) console.log(JSON.stringify(config.output.areas[cnt].points));
			var areaSquarePixels = area(config.output.areas[cnt].points);
			totalStickerSquarePixels += areaSquarePixels;
		}
		
	}

	if(totalStickerSquarePixels != 0) {
		//Only override the auto-calculated option if there are sticker entries (i.e. not 0)
		config.output.stickerSquarePixels = totalStickerSquarePixels;
	}


	var woundLayer = [];
	var underLayer = [];

	for(var cnt = 0; cnt< config.output.areas.length; cnt++) {

		//Finds the areas in square pixels and square cm.
		if(config.output.areas[cnt].points) {
			if(verbose == true) console.log("Calcing area of:" + JSON.stringify(config.output.areas[cnt].points));
			var areaSquarePixels = area(config.output.areas[cnt].points);
			if(verbose == true) console.log("Area:" + areaSquarePixels);
		} else {
			var areaSquarePixels = 0;
		}


		config.output.areas[cnt].areaSquarePixels = areaSquarePixels;
		if(config.output.stickerSquarePixels) {
			var areasSquareCm = (areaSquarePixels / config.output.stickerSquarePixels) * config.input.sticker.physicalAreaSquareCm;
			config.output.areas[cnt].areaSquareCm = areasSquareCm.toFixed(3);
		} else {
			config.output.areas[cnt].areaSquareCm = 0;
		}

		//Set in the config file
		if(config.output.areas[cnt].type == "wound") {
			totalWoundSquarePixels += areaSquarePixels;
			totalWoundSquareCm += areasSquareCm;
			woundLayer.push(cnt);
		}
		
		if(config.output.areas[cnt].type == "under") {
			totalUnderSquarePixels += areaSquarePixels;
			totalUnderSquareCm += areasSquareCm;
			underLayer.push(cnt);
		}
		
		
		if(config.output.areas[cnt].type == "negative") {
			totalWoundSquarePixels -= areaSquarePixels;
			totalWoundSquareCm -= areasSquareCm;
		}
		
	}

	config.output.fullWound.areaSquarePixels = totalWoundSquarePixels;
	if((totalWoundSquareCm == 0)||(totalStickerSquarePixels == 0)) {
		config.output.fullWound.areaSquareCm = "[Unknown]";
		
	} else {
		config.output.fullWound.areaSquareCm = totalWoundSquareCm.toFixed(3);
	}
	
	try {
		if((totalUnderSquarePixels > 0)&&(totalStickerSquarePixels != 0)) {
			//We have some undermining - also set the wound version of these, and then create a total

			//First need to re-calculate the under area based on subtracting any area from the wound area that overlaps (i.e. the intersection with the wound area)
			if((totalWoundSquarePixels !=0) && (totalWoundSquarePixels > 0)) {
				//Find the intersection

				//Convert arrays
				var poly1 = [];
				var poly2 = [];
				var woundArray = [];
				var underArray = [];
				if(verbose == true) console.log("Wound layer: " + woundLayer + " Under Layer: " + underLayer);
			
				//Add all of the polygons to the master polys
				for(var woundPoly = 0; woundPoly < woundLayer.length; woundPoly++) {
					woundArray.push(config.output.areas[woundLayer[woundPoly]].points);
				}
				for(var underPoly = 0; underPoly < underLayer.length; underPoly++) {
					underArray.push(config.output.areas[underLayer[underPoly]].points);
				}
			
				var poly1 = polygon(woundArray);
				var poly2 = polygon(underArray);
			

				poly1.geometry.type = "Polygon";
				poly2.geometry.type = "Polygon";
				if(verbose == true) console.log("Poly1:" + JSON.stringify(poly1) + " \n\nPoly2:" + JSON.stringify(poly2));

			
					var intersectedPoly = intersect(poly1, poly2);
			
					if(intersectedPoly) {
						if(verbose == true) console.log("Intersected Poly:" + JSON.stringify(intersectedPoly));
				
						var outIntersectedPoly = [];
						var intersectedAreaSqPixels = 0;
				
						if(intersectedPoly.geometry.coordinates.length > 1) {
							//A multi polygon
							for(var poly = 0; poly < intersectedPoly.geometry.coordinates.length; poly ++) { 
								outIntersectedPoly[poly] = [];
								for(var lcnt = 0; lcnt < intersectedPoly.geometry.coordinates[poly][0].length; lcnt++) {
									outIntersectedPoly[poly][lcnt] = { x: intersectedPoly.geometry.coordinates[poly][0][lcnt][0],
											y: intersectedPoly.geometry.coordinates[poly][0][lcnt][1] };

								}
						
								if(verbose == true) console.log("Intersected before area poly:" + JSON.stringify(outIntersectedPoly[poly]));

						
								var thisArea = area(outIntersectedPoly[poly]);
								if(thisArea) {
									intersectedAreaSqPixels += thisArea;
								}
							}
						} else {
							//Single polygon
							outIntersectedPoly = [];
							for(var lcnt = 0; lcnt < intersectedPoly.geometry.coordinates[0].length; lcnt++) {
								outIntersectedPoly[lcnt] = { x: intersectedPoly.geometry.coordinates[0][lcnt][0],
										y: intersectedPoly.geometry.coordinates[0][lcnt][1] };

							}
					
							if(verbose == true) console.log("Intersected before area poly:" + JSON.stringify(outIntersectedPoly));

					
							var thisArea = area(outIntersectedPoly);
							if(thisArea) {
								intersectedAreaSqPixels += thisArea;
							}
				
						}

				

						if(verbose == true) console.log("Intersected area pixels:" + intersectedAreaSqPixels);
						totalUnderSquarePixels = totalUnderSquarePixels - intersectedAreaSqPixels;
				
						totalUnderSquareCm = (totalUnderSquarePixels / config.output.stickerSquarePixels) * config.input.sticker.physicalAreaSquareCm;
					}
			
			}

			config.output.fullWound.areaSquarePixels = totalWoundSquarePixels + totalUnderSquarePixels;
			config.output.fullWound.areaSquareCm = totalWoundSquareCm +  totalUnderSquareCm;
			config.output.fullWound.areaSquareCm = config.output.fullWound.areaSquareCm.toFixed(3);		//Ensure 3 dp string
		
		
			config.output.fullWound.woundAreaSquarePixels = totalWoundSquarePixels;
			config.output.fullWound.woundAreaSquareCm = totalWoundSquareCm.toFixed(3);
		
			config.output.fullWound.underAreaSquarePixels = totalUnderSquarePixels;
			config.output.fullWound.underAreaSquareCm = totalUnderSquareCm.toFixed(3);
		} else {
			//Otherwise clear these off
			config.output.fullWound.woundAreaSquarePixels = "";
			config.output.fullWound.woundAreaSquareCm = "";
		
			config.output.fullWound.underAreaSquarePixels = "";
			config.output.fullWound.underAreaSquareCm = "";
		
		}
	
	}		//End of intersected poly try
	catch(err) {		//There was a problem with the saving process
		config.output.fullWound.woundAreaSquarePixels = "";
		config.output.fullWound.woundAreaSquareCm = "";
		
		config.output.fullWound.underAreaSquarePixels = "";
		config.output.fullWound.underAreaSquareCm = "";
		config.output.fullWound.areaSquareCm = "[Unknown - Error]";
	}



	return config;
}




module.exports.calculateIntersectionArea = function(config)
{
	
	return config;
}


module.exports.refreshThumbnail = function(cv, photoFile, webPhotoFile, conf, cb)
{
	//Start with the larger web-view version of the image, draw transparent polygons around
	//the wound polygons, and polygons around the sticker in a different colour,
	//and resize the image to thumbnail size and write out over the top of the last thumbnail.
	//This function is called from both a wound generation and after a save operation.


	// (B)lue, (G)reen, (R)ed, Transparency
	var stickerColour = [0, 255, 0, 90];		//Was 50 transparency		
	var woundColour = [0, 0, 255, 90];
	var underColour = [43, 80, 98, 90];		//rgb(98, 80, 43)
	var negativeColour = [98, 48, 80, 90];  //RGB: 80, 43, 98
	var freestyleColour = [255, 255, 255, 90];  //RGB: 240, 240, 240
	var lineThickness = 3;


	/*var GREEN = [0, 255, 0]; // B, G, R
		var WHITE = [255, 255, 255]; // B, G, R
		var BLACK = [0, 0, 0]; // B, G, R
		var RED   = [0, 0, 255]; // B, G, R
		var BRIGHTRED = [153, 153, 255]; // B, G, R */

	if(verbose == true) console.log("Reading web photos file: " + webPhotoFile);

	//Read in the 1200 pixel photo file
	// load local image file with jimp. It supports jpg, png, bmp, tiff and gif:
  	jimp.read(webPhotoFile, function(err, jimpIm) {
 		
	    if (err) throw new Error("Error reading photo:" + webPhotoFile + " Error:" + err);

 		// `jimpImage.bitmap` property has the decoded ImageData that we can use to create a cv:Mat
 	    var webPhoto = cv.matFromImageData(jimpIm.bitmap);
 	      
 	      
	    var width = webPhoto.cols;
		var height = webPhoto.rows;



		if (width < 1 || height < 1) throw new Error("Sorry, the image " + webPhotoFile + " needs to have a valid size. Width =" + width + " Height=" + height);

		var polyBuff = cv.Mat.zeros(webPhoto.rows, webPhoto.cols, webPhoto.type());

		//Go through each polygon area
		var opencvPolySticker = [];
		var opencvPolyWound = [];
		var opencvPolyUnder = [];
		var opencvPolyNegative = [];
		var opencvPolyFreestyle = [];
		if(!conf.output.areas) conf.output.areas = [];



		for(var cnt = 0; cnt< conf.output.areas.length; cnt++) {
			var points = conf.output.areas[cnt].points;

			var arrayHere = [];
			var fromXY = [ points[0][0], height - points[0][1] ];		//Get first point
			var firstXY = fromXY;
			
			var ignore = false;
			var freestyleCheck = false;	

			switch(conf.output.areas[cnt].type) {
				
				case "sticker":
					var sticker = true;
					var colour = stickerColour;
					var thislineThickness = lineThickness;
				break;
				
				case "wound":
					var sticker = false;
					var colour = woundColour;
					var thislineThickness = lineThickness;
				break;
				
				case "under":
					var sticker = false;
					var colour = underColour;
					var thislineThickness = lineThickness;
				break;
				
				case "negative":
					var sticker = false;
					var colour = negativeColour;
					var thislineThickness = lineThickness;
				break;
				
				case "freestyle":
					var sticker = false;
					var colour = freestyleColour;
					var thislineThickness = 10;
					freestyleCheck = true;		
				break;	
					
				default:
					ignore = true;
				break;
			
			}


			if(ignore == false) {


				for(var pcnt = 0; pcnt < points.length; pcnt++) {
					var point = [ points[pcnt][0], height - points[pcnt][1] ];
					toXY = point;

					
					
					
					fromXY = toXY;
					
					arrayHere.push(point);
					
					//See https://stackoverflow.com/questions/56370649/create-contour-in-opencv-js
					
				}
				
				
				//let points = [[10,10],[100,10],[100,100],[10,100]];
				var outputPoints = [];
				for(var pcnt = 0; pcnt < points.length; pcnt++) {
					var point = [ points[pcnt][0], height - points[pcnt][1] ];
					outputPoints.push(point);
				}
				
				
				
					
				var theseContours = new cv.MatVector();
				let mat = cv.matFromArray(1, points.length, cv.CV_32SC2, outputPoints.flat(2));
				theseContours.push_back(mat);
				
				if(freestyleCheck) {
					//Just draw polylines, not the enclosed contour
					cv.polylines(polyBuff, theseContours, false, colour, thislineThickness);
				} else {
					//Draw the fully closed contours
					var hierarchy = new cv.Mat();							
					cv.drawContours(polyBuff, theseContours, -1, colour, thislineThickness, cv.LINE_8);		//, hierarchy, 100
				}
			}

	
			switch(conf.output.areas[cnt].type) {
				
				case "sticker":
					opencvPolySticker.push(arrayHere);
				break;
				
				case "wound":
					opencvPolyWound.push(arrayHere);
				break;
				
				case "under":
					opencvPolyUnder.push(arrayHere);
				break;
				
				case "negative":
					opencvPolyNegative.push(arrayHere);
				break;
				
				case "freestyle":
					opencvPolyFreestyle.push(arrayHere);
				break;
			
			}

			


		}
		if(verbose == true) console.log("Wound polys on thumbnail:" + JSON.stringify(opencvPolyWound));
		if(verbose == true) console.log("Sticker polys on thumbnail:" + JSON.stringify(opencvPolySticker));
		if(opencvPolyWound) {
			//No need for a red background polygon: polyBuff.fillPoly(opencvPolyWound, woundColour);
		}
		if(opencvPolySticker) {
			//Make yellow stickers green - like a traffic light!
			//Only show outlines now. Was: polyBuff.fillPoly(opencvPolySticker, stickerColour);
		}
		if(opencvPolyUnder) {
			//No need for a brown background polygon: polyBuff.fillPoly(opencvPolyUnder, underColour);
		}
		if(opencvPolyNegative) {
			//Only show outlines now. Was: polyBuff.fillPoly(opencvPolyNegative, negativeColour);
		}
		
		if(opencvPolyFreestyle) {
			//Only show outlines now. 
		}

		var outBuff = webPhoto.clone();

		//Now blend in with some alpha transparency the buffer to the webphoto
		//outBuff.addWeighted(webPhoto, 0.15, polyBuff, 0.85);  //0.8 to 0.2 was good for high vis.
		cv.addWeighted(webPhoto, 0.7, polyBuff, 0.3, 0, outBuff);


		 //And create a thumbnail
		  delete webPhoto;
		  delete outBuff;
		  var thumbnailPhotoFile = photoFile.replace(conf.output.scaledPhoto.nameConvention.replaceText,
												conf.output.scaledPhoto.nameConvention.withThumbnailText);
			
		  if(verbose == true) console.log('Saving thumbnail photo...' + thumbnailPhotoFile);	
		  
		  new jimp({
					width: outBuff.cols,
					height: outBuff.rows,
					data: Buffer.from(outBuff.data)
				  })
				  .resize(conf.output.scaledPhoto.thumbnailWidth, conf.output.scaledPhoto.thumbnailHeight) // resize
				  .quality(95) // set JPEG quality
				  .write(thumbnailPhotoFile);


		 
		  delete thumbnailPhoto;
		  if(verbose == true) console.log('Done');

		  if(verbose == true) console.log('All images saved.');

		  cb(null, thumbnailPhotoFile);
	})

	return;
}


